//=================================================================
//                      ChartSeriesColors.java
//=================================================================
//
//    This class provides the basic ScatterPlot series colors and
// the ability to modifiy those colors.
//
//                   << ChartSeriesColors.java >>
//
//=================================================================
// Copyright (C) 2017 Dana M. Proctor
// Version 1.2 11/21/2017
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version
// 2 of the License, or (at your option) any later version. This
// program is distributed in the hope that it will be useful, 
// but WITHOUT ANY WARRANTY; without even the implied warranty
// of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See
// the GNU General Public License for more details. You should
// have received a copy of the GNU General Public License along
// with this program; if not, write to the Free Software Foundation,
// Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
// (http://opensource.org)
//=================================================================
// Revision History
// Changes to the code should be documented here and reflected
// in the present version number. Author information should
// also be included with the original copyright author.
//=================================================================
// Version 1.0 10/02/2017 Original ChartSeriesColors Class.
//         1.1 10/05/2017 Constructor Changed Assignment of paintSequence to
//                        New private Method createPaintArray(). Changed Method
//                        createDefaultPaintArray() to Public, Static, No alpha
//                        to seriesColors.
//         1.2 11/21/2017 Modified to be Used as Demostration Plugin for lindyFrame.
//
//-----------------------------------------------------------------
//                 danap@dandymadeproductions.com
//=================================================================

package com.dandymadeproductions.scatterplot.data;

import java.awt.Color;
import java.awt.Paint;
import java.awt.Shape;
import java.awt.Stroke;

import org.jfree.chart.ChartColor;
import org.jfree.chart.plot.DefaultDrawingSupplier;

import com.dandymadeproductions.scatterplot.ScatterPlot;
import com.dandymadeproductions.scatterplot.charts.ChartProperties;

/**
 *    The DataChartSeriesColors class provides the basic DataCharts series
 * colors and the ability to modifiy those colors.
 * 
 * @author Dana M. Proctor
 * @version 1.2 11/21/2017
 */

public class ChartSeriesColors
{
   // Class Instances
   private DefaultDrawingSupplier drawingSupplier;

   private Paint[] paintSequence;
   private Paint[] outlinePaintSequence;
   private Paint[] fillPaintSequence;
   private Stroke[] strokeSequence;
   private Stroke[] outlineStrokeSequence;
   private Shape[] shapeSequence;

   private static final int PAINT_SEQUENCE_ALPHA =  0xFF;

   //==============================================================
   // ChartSeriesColors Constructors
   //==============================================================

   public ChartSeriesColors()
   {
      this(PAINT_SEQUENCE_ALPHA);
   }

   public ChartSeriesColors(int alpha)
   {
      paintSequence = createPaintArray(alpha);
      outlinePaintSequence = DefaultDrawingSupplier.DEFAULT_OUTLINE_PAINT_SEQUENCE;
      fillPaintSequence = DefaultDrawingSupplier.DEFAULT_FILL_PAINT_SEQUENCE;
      strokeSequence = DefaultDrawingSupplier.DEFAULT_STROKE_SEQUENCE;
      outlineStrokeSequence = DefaultDrawingSupplier.DEFAULT_OUTLINE_STROKE_SEQUENCE;
      shapeSequence = DefaultDrawingSupplier.DEFAULT_SHAPE_SEQUENCE;

      drawingSupplier = new DefaultDrawingSupplier(paintSequence, outlinePaintSequence, fillPaintSequence,
                                                   strokeSequence, outlineStrokeSequence, shapeSequence);
   }
   
   //==============================================================
   // Method to setup the default, stored modified sequence with
   // alpha.
   //==============================================================
   
   private Paint[] createPaintArray(int alpha)
   {
      // Method Instances
      ChartProperties chartProperties;
      Paint[] seriesColors;
      int red, green, blue;

      // Setup Basis
      chartProperties = ScatterPlot.getChartDefaultProperties();
      seriesColors = ChartColor.createDefaultPaintArray();
      
      // Add alpha
      for (int i = 0; i < seriesColors.length; i++)
      {
         if (i > ChartProperties.PLOT_SERIES_SEQUENCE_NUMBER)
         {
            red = ((Color) seriesColors[i]).getRed();
            green = ((Color) seriesColors[i]).getGreen();
            blue = ((Color) seriesColors[i]).getBlue();
            seriesColors[i] = new Color(red, green, blue, alpha);
         }    
      }
      
      // Modified Default Sequence
      for (int i = 0; i < ChartProperties.PLOT_SERIES_SEQUENCE_NUMBER; i++)
         seriesColors[i] = new Color(chartProperties.getPlotSeriesColor(i).getRed(),
                                     chartProperties.getPlotSeriesColor(i).getGreen(),
                                     chartProperties.getPlotSeriesColor(i).getBlue(),
                                     alpha);

      return seriesColors;
   }

   //==============================================================
   // Method to setup the default paint series sequence.
   //==============================================================

   public static Paint[] createDefaultPaintArray()
   {
      // Method Instances
      Paint[] seriesColors;

      // Collect Default
      seriesColors = ChartColor.createDefaultPaintArray();
      
      // Modified Default Sequence

      // Orange
      seriesColors[0] = new Color(0xFF, 0x80, 0x00);
      // Green
      seriesColors[1] = new Color(0x55, 0xFF, 0x55);
      // Blue
      seriesColors[2] = new Color(0x55, 0x55, 0xFF);
      // Yellow
      seriesColors[3] = new Color(0xFF, 0xFF, 0x55);
      // Violet
      seriesColors[4] = new Color(0xC6, 0x0B, 0xFF);
      // Red
      seriesColors[5] = new Color(0xFF, 0x55, 0x55);
      // Aqua
      seriesColors[6] = new Color(0x55, 0xFF, 0xFF);

      return seriesColors;
   }
   
   //==============================================================
   // Method to setup the default paint series sequence.
   //==============================================================

   public DefaultDrawingSupplier getDefaultDrawingSupplier()
   {
      return drawingSupplier;
   }
}