//=================================================================
//                       ScatterChartPanel.java
//=================================================================
//
//    This class provides the essentials to create a JFreeChart
// Scatter Chart for the lindyFrame Plugin ScatterPlot.
//
//                    << ScatterChartPanel.java >>
//
//=================================================================
// Copyright (C) 2017 Dana M. Proctor
// Version 1.1 10/01/2017
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version
// 2 of the License, or (at your option) any later version. This
// program is distributed in the hope that it will be useful, 
// but WITHOUT ANY WARRANTY; without even the implied warranty
// of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See
// the GNU General Public License for more details. You should
// have received a copy of the GNU General Public License along
// with this program; if not, write to the Free Software Foundation,
// Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
// (http://opensource.org)
//=================================================================
// Revision History
// Changes to the code should be documented here and reflected
// in the present version number. Author information should
// also be included with the original copyright author.
//=================================================================
// Version 1.0 09/24/2017 Original ScatterChartPanel Class.
//         1.1 10/01/2017 Method createChart() Moved Setting Default Properties to
//                        Parent, ChartsPanel.setDefaultProperties(). Removed Class
//                        Instance chartDefaultProperties.
//
//-----------------------------------------------------------------
//                 danap@dandymadeproductions.com
//=================================================================

package com.dandymadeproductions.scatterplot.charts;

import javax.swing.JOptionPane;

import org.jfree.chart.ChartFactory;
import org.jfree.chart.ChartPanel;
import org.jfree.chart.plot.XYPlot;
import org.jfree.chart.renderer.xy.XYItemRenderer;
import org.jfree.chart.renderer.xy.XYLineAndShapeRenderer;
import org.jfree.data.xy.XYSeries;
import org.jfree.data.xy.XYSeriesCollection;

import com.dandymadeproductions.scatterplot.ScatterPlot;


/**
 *    The ScatterChartPanel class provides the essentials to create a
 * JFreeChart Scatter Chart for the lindyFrame Plugin ScatterPlot.
 * 
 * @author Dana M. Proctor
 * @version 1.1 10/01/2017
 */

public class ScatterChartPanel extends ChartsPanel
{
   private static final long serialVersionUID = -3465778000758307134L;
   
   // Class Instances
   private XYSeriesCollection xyDataSet;
   
   private static int SET_EXAMPLE_DATA_NUMBER = 3;
   
   //==============================================================
   // ScatterChartPanel Constructor
   //==============================================================
   
   public ScatterChartPanel(String resourceExample, String resourceChart)
   {
      this.resourceExample = resourceExample;
      this.resourceChart = resourceChart;
      
      chartDefaultProperties = ScatterPlot.getChartDefaultProperties();
      
      add(createExampleChart());
   }
   
   //==============================================================
   // Class method to provide an example chart through fixed data
   // to demostrate the use of the Scatter Chart.
   // 
   // Note: X-Axis, Domain - Number
   //       Y-Axis, Range - Number
   //==============================================================
   
   private ChartPanel createExampleChart()
   {
      // Method Instances
      XYSeries xySeries;
      
      xyDataSet = new XYSeriesCollection();
      xySeries = new XYSeries("Series 1", false, true);
      
      // Add Example Data
      for (int j = 0; j < 30; j++)
         xySeries.add(Math.random() * 98, Math.random() * 98);
      
      xyDataSet.addSeries(xySeries);
      
      // Create Scatter Chart
      createChart("\t\t\t\t" + resourceExample + " Scatter " + resourceChart + "\n\n"
                         + "SELECT (X-AXIS) Number, (Y-AXIS) Number, FROM TABLE");
      
      chartPanel = new ChartPanel(chart);
      return chartPanel;
   }
   
   //==============================================================
   // Class method to actually create the chart and set basic
   // characteristics.
   //==============================================================
   
   private void createChart(String title)
   {
      // Method Instances
      XYItemRenderer xyRenderer;
      XYLineAndShapeRenderer shapeRenderer;
      
      chart = ChartFactory.createScatterPlot(title, "X-AXIS", "Y-AXIS", xyDataSet);
      
      // Set Properties
      
      // Default
      setProperties(ScatterPlot.SCATTER_CHART);
      
      // XY Chart Specific
      xyRenderer = ((XYPlot) chart.getPlot()).getRenderer();
      
      if (xyRenderer instanceof XYLineAndShapeRenderer)
      {
         shapeRenderer = (XYLineAndShapeRenderer) xyRenderer;
         shapeRenderer.setBaseShapesFilled(chartDefaultProperties.getPlotBaseShapeFilled());
      }
   }
   
   //==============================================================
   // Class methods to provide the ability to return the chart
   // title and properties.
   //==============================================================
   
   public String getChartTitle()
   {
      return chart.getTitle().getText();
   }
   
   public void setChartTitle(String value)
   {
      chart.setTitle(value);
   }
   
   public String getChartXAxisLabel()
   {
      return chart.getXYPlot().getDomainAxis().getLabel();
   }
   
   public void setChartXAxisLabel(String value)
   {
      chart.getXYPlot().getDomainAxis().setLabel(value);
   }
   
   public String getChartYAxisLabel()
   {
      return chart.getXYPlot().getRangeAxis().getLabel();
   }
   
   public void setChartYAxisLabel(String value)
   {
      chart.getXYPlot().getRangeAxis().setLabel(value);
   }
   
   //==============================================================
   // Class methods to set the example data series to the scatter
   // chart.
   //==============================================================
   
   public void setExampleData()
   {
      // Method Instances
      XYSeries xySeries;
      
      // Remove current series data.
      xyDataSet.removeAllSeries();
      
      // Set Example Data
      for (int i = 0; i < SET_EXAMPLE_DATA_NUMBER; i++)
      {
         xySeries = new XYSeries("Series " + i, false, true);
         
         double x = 0;
         for (int j = 0; j < 50; j++)
         {
            x += Math.random() * 100;
            xySeries.add(x, Math.random() * 800);
         }
         xyDataSet.addSeries(xySeries);
      }
      
      // Reset Title
      chart.setTitle(resourceExample + " Scatter " + resourceChart);
   }
   
   //==============================================================
   // Class methods to add, clear, remove all data, and collect
   // series count from the existing chart.
   //==============================================================
   
   public void addSeries(XYSeries xySeries)
   {  
      // Add Series.
      try
      {
         if (xyDataSet == null)
            xyDataSet = new XYSeriesCollection();
         
         xyDataSet.addSeries(xySeries);
      }
      catch (IllegalArgumentException iae)
      {
         JOptionPane.showMessageDialog(null, iae.toString(), "Alert", JOptionPane.ERROR_MESSAGE);
      }
   }
   
   public void clearData()
   {
      xyDataSet.removeAllSeries();
   }
   
   public void removeAllData()
   {
      // Insert new chart to insure all aspects of chart
      // may be modified.
      
      removeAll();
      xyDataSet = new XYSeriesCollection();
      createChart(" ");
      
      chartPanel = new ChartPanel(chart);
      add(chartPanel);
      validate();
   }
   
   public int getSeriesCount()
   {
      return xyDataSet.getSeriesCount();
   }
}