//=================================================================
//                       LineChartPanel.java
//=================================================================
//
//    This class provides the essentials to create a JFreeChart
// Line Chart for the lindyFrame Plugin ScatterPlot.
//
//                    << LineChartPanel.java >>
//
//=================================================================
// Copyright (C) 2017 Dana M. Proctor
// Version 1.7 11/20/2017
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version
// 2 of the License, or (at your option) any later version. This
// program is distributed in the hope that it will be useful, 
// but WITHOUT ANY WARRANTY; without even the implied warranty
// of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See
// the GNU General Public License for more details. You should
// have received a copy of the GNU General Public License along
// with this program; if not, write to the Free Software Foundation,
// Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
// (http://opensource.org)
//=================================================================
// Revision History
// Changes to the code should be documented here and reflected
// in the present version number. Author information should
// also be included with the original copyright author.
//=================================================================
// Version 1.0 09/13/2017 Original LineChartPanel Class.
//         1.1 09/19/2017 Organized Imports. Method removeAllData() Added Space
//                        for chart Title. Added Method getSeriesCount(). Method
//                        addSeries() Cleaned Up, Setup for Adding Controlled
//                        Series and try, catch for Duplicate Series, Keys.
//         1.2 09/20/2017 Made Method createExampleLineChart() private. Added
//                        Method createChart() & Used to Create Chart & Set
//                        Defaults.
//         1.3 09/22/2017 Added Class Instance chartDefaultProperties() & Used
//                        in createChart() to Replace Static Basic Properties.
//                        Method addSeries() Removed Instances freeMemory, dataSetSize,
//                        & estimatedMemoryRequirements & Their Use to Control
//                        Old Methods addControlledSeries(). Cleanup & Comment
//                        Changes.
//         1.4 09/22/2017 Removed Constructor Arugument queryStatementString.
//         1.5 09/30/2017 Removed Class Instance chartDefaultProperties. Changed
//                        createLineExampleLineChart() createExampleChart(). Added
//                        in createExampleChart() Setting Defaults for Paint Title,
//                        Labels, Axis, & Ticks. Comments.
//         1.6 10/01/2017 Method createChart() Moved Setting Default Properties to
//                        Parent, ChartsPanel.setDefaultProperties().
//         1.7 11/20/2017 Original ScatterPlot LineChartPanel Class.
//
//-----------------------------------------------------------------
//                 danap@dandymadeproductions.com
//=================================================================

package com.dandymadeproductions.scatterplot.charts;

import javax.swing.JOptionPane;

import org.jfree.chart.ChartFactory;
import org.jfree.chart.ChartPanel;
import org.jfree.chart.plot.XYPlot;
import org.jfree.chart.renderer.xy.XYItemRenderer;
import org.jfree.chart.renderer.xy.XYLineAndShapeRenderer;
import org.jfree.data.xy.XYSeries;
import org.jfree.data.xy.XYSeriesCollection;

import com.dandymadeproductions.scatterplot.ScatterPlot;

/**
 *    The LineChartPanel class provides the essentials to create a
 * JFreeChart Line Chart for the lindyFrame Plugin ScatterPlot.
 * 
 * @author Dana M. Proctor
 * @version 1.7 11/20/2017
 */

public class LineChartPanel extends ChartsPanel
{
   private static final long serialVersionUID = 6758153488799132631L;

   // Class Instances
   private XYSeriesCollection xyDataSet;
   
   private static int SET_EXAMPLE_DATA_NUMBER = 3;
   
   //==============================================================
   // LineChartPanel Constructor
   //==============================================================
   
   public LineChartPanel(String resourceExample, String resourceChart)
   {
      this.resourceExample = resourceExample;
      this.resourceChart = resourceChart;
      
      add(createExampleChart());
   }
   
   //==============================================================
   // Class method to provide an example chart through fixed data
   // to demostrate the use of the Line Chart.
   // 
   // Note: X-Axis, Domain - Number
   //       Y-Axis, Range - Number
   //==============================================================
   
   private ChartPanel createExampleChart()
   {
      // Method Instances
      XYSeries xySeries;
      
      xyDataSet = new XYSeriesCollection();
      xySeries = new XYSeries("Series 1", false, true);
      
      double x = 0;
      for (int j = 0; j < 5; j++)
      {
         x += Math.random() * 100;
         xySeries.add(x, Math.random() * 800);
      }
      xyDataSet.addSeries(xySeries);
      
      // Create XY Line Chart
      createChart("\t\t\t\t" + resourceExample + " Line " + resourceChart + "\n\n"
                         + "SELECT (X-AXIS) Number, (Y-AXIS) Number, FROM TABLE");
      
      chartPanel = new ChartPanel(chart);
      return chartPanel;
   }
   
   //==============================================================
   // Class method to actually create the chart and set basic
   // characteristics.
   //==============================================================
   
   private void createChart(String title)
   {
      // Method Instances
      XYItemRenderer xyRenderer;
      XYLineAndShapeRenderer shapeRenderer;
      
      chart = ChartFactory.createXYLineChart(title, "X-AXIS", "Y-AXIS", xyDataSet);
      
      // Set Properties
      
      // Default
      setProperties(ScatterPlot.LINE_CHART);
      
      // XY Chart Specific
      xyRenderer = ((XYPlot) chart.getPlot()).getRenderer();
      
      if (xyRenderer instanceof XYLineAndShapeRenderer)
      {
          shapeRenderer = (XYLineAndShapeRenderer) xyRenderer;
          shapeRenderer.setBaseShapesVisible(chartDefaultProperties.getPlotBaseShapeVisible());
          shapeRenderer.setBaseShapesFilled(chartDefaultProperties.getPlotBaseShapeFilled());
          shapeRenderer.setDrawSeriesLineAsPath(chartDefaultProperties.getDrawSeriesLinePath());
      }
   }
   
   //==============================================================
   // Class methods to provide the ability to return the chart
   // title and properties.
   //==============================================================
   
   public String getChartTitle()
   {
      return chart.getTitle().getText();
   }
   
   public void setChartTitle(String value)
   {
      chart.setTitle(value);
   }
   
   public String getChartXAxisLabel()
   {
      return chart.getXYPlot().getDomainAxis().getLabel();
   }
   
   public void setChartXAxisLabel(String value)
   {
      chart.getXYPlot().getDomainAxis().setLabel(value);
   }
   
   public String getChartYAxisLabel()
   {
      return chart.getXYPlot().getRangeAxis().getLabel();
   }
   
   public void setChartYAxisLabel(String value)
   {
      chart.getXYPlot().getRangeAxis().setLabel(value);
   }
   
   //==============================================================
   // Class methods to set the example data series to the line
   // chart.
   //==============================================================
   
   public void setExampleData()
   {
      // Method Instances
      XYSeries xySeries;
      
      // Remove current series data.
      xyDataSet.removeAllSeries();
      
      // Set Example Data
      for (int i = 0; i < SET_EXAMPLE_DATA_NUMBER; i++)
      {
         xySeries = new XYSeries("Series " + i, false, true);
         
         double x = 0;
         for (int j = 0; j < 5; j++)
         {
            x += Math.random() * 100;
            xySeries.add(x, Math.random() * 800);
         }
         xyDataSet.addSeries(xySeries);
      }
      
      // Reset Title
      chart.setTitle(resourceExample + " Line " + resourceChart);
   }
   
   //==============================================================
   // Class methods to add, clear, remove all data, and collect
   // series count from the existing chart.
   //==============================================================
   
   public void addSeries(XYSeries xySeries)
   {  
      // Add Series.
      try
      {
         if (xyDataSet == null)
            xyDataSet = new XYSeriesCollection();
         
         xyDataSet.addSeries(xySeries);
      }
      catch (IllegalArgumentException iae)
      {
         JOptionPane.showMessageDialog(null, iae.toString(), "Alert", JOptionPane.ERROR_MESSAGE);
      }
   }
   
   public void clearData()
   {
      xyDataSet.removeAllSeries();
   }
   
   public void removeAllData()
   {
      // Insert new chart to insure all aspects of chart
      // may be modified.
      
      removeAll();
      xyDataSet = new XYSeriesCollection();
      createChart(" ");
      
      chartPanel = new ChartPanel(chart);
      add(chartPanel);
      validate();
   }
   
   public int getSeriesCount()
   {
      return xyDataSet.getSeriesCount();
   }
}