//=================================================================
//                 ChartProperties Class
//=================================================================
//    This class provides a structure properties storage for the
// charts in the ScatterPlot.
//
//                << ChartProperties.java >>
//
//=================================================================
// Copyright (C) 2017 Dana M. Proctor
// Version 1.4 11/20/2017
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version
// 2 of the License, or (at your option) any later version. This
// program is distributed in the hope that it will be useful, 
// but WITHOUT ANY WARRANTY; without even the implied warranty
// of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See
// the GNU General Public License for more details. You should
// have received a copy of the GNU General Public License along
// with this program; if not, write to the Free Software Foundation,
// Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
// (http://opensource.org)
//
//=================================================================
// Revision History
// Changes to the code should be documented here and reflected
// in the present version number. Author information should
// also be included with the original copyright author.
//=================================================================
// Version 1.0 Original ChartDefaultProperties Class.
//         1.1 Added Additional Property plotBackgroundColor.
//         1.2 Added Class Instances plotSeriesColors & DATACHARTS_PLOTSERIESCOLORS,
//             for Property of Same. Some Formatting for Clarity.
//         1.3 Changed Name to ChartProperties.
//         1.4 Corrected Package After Move to charts Directory. Added Import,
//             ChartSeriesColors.
//
//-----------------------------------------------------------------
//                 danap@dandymadeproductions.com
//=================================================================

package com.dandymadeproductions.scatterplot.charts;

import java.awt.Color;
import java.awt.Paint;

import com.dandymadeproductions.lindyframe.LindyFrame;
import com.dandymadeproductions.scatterplot.data.ChartSeriesColors;

/**
 *    The ChartProperties class provides a default structure properties
 * storage for the charts in the ScatterPlot.
 * 
 * @author Dana M. Proctor
 * @version 1.4 11/20/2017
 */

public class ChartProperties
{
   // Class Instances.
   
   // Chart
   private int titleFontSize;
   private int chartForegroundColor;
   private int chartBackgroundColor;
   
   // Series
   private boolean autoSort;
   private boolean allowDuplicateXValues;
   
   // Plot
   private int plotBackgroundColor;
   private int domainGridLineColor;
   private int rangeGridLineColor;
   private boolean plotBaseShapeVisible;
   private boolean plotBaseShapeFilled;
   private int[] plotSeriesColors;
   
   // Line Charts
   private boolean drawSeriesLinePath;
   
   public static final int PLOT_SERIES_SEQUENCE_NUMBER = 7;
   
   public static final String DATACHARTS_TITLEFONTSIZE = "DataChartsTitleFontSize";
   public static final String DATACHARTS_CHARTFOREGROUNDCOLOR = "DataChartsChartForegroundColor";
   public static final String DATACHARTS_CHARTBACKGROUNDCOLOR = "DataChartsChartBackgroundColor";
   
   public static final String DATACHARTS_AUTOSORT = "DataChartsAutoSort";
   public static final String DATACHARTS_ALLOWDUPLICATEXVALUES = "DataChartsAllowDuplicateXValues";
   
   public static final String DATACHARTS_PLOTBACKGROUNDCOLOR = "DataChartPlotBackgroundColor";
   public static final String DATACHARTS_DOMAINGRIDLINECOLOR = "DataChartsDomainGridLineColor";
   public static final String DATACHARTS_RANGEGRIDLINECOLOR = "DataChartsRangeGridLineColor";
   public static final String DATACHARTS_PLOTBASESHAPEVISIBLE = "DataChartsBaseShapeVisibile";
   public static final String DATACHARTS_PLOTBASESHAPEFILLED = "DataChartsBaseShapeFilled";
   public static final String DATACHARTS_PLOTSERIESCOLORS = "DataChartsPlotSeriesColors";
   
   public static final String DATACHARTS_DRAWSERIESLINEPATH = "DataChartsDrawSeriesLinePath";
   
   //==============================================================
   // ChartDefaultProperties Constructor
   //==============================================================

   public ChartProperties()
   {
      // Set Default State.
      
      // Chart
      titleFontSize = LindyFrame.getGeneralProperties().getFontSize() + 2;
      chartForegroundColor = ChartsPanel.DEFAULT_FOREGROUND.getRGB();
      chartBackgroundColor = ChartsPanel.DEFAULT_BACKGROUND.getRGB();
      
      // Series
      autoSort = ChartsPanel.DEFAULT_AUTOSORT;
      allowDuplicateXValues = ChartsPanel.DEFAULT_ALLOWDUPLICATEXVALUES;
      
      // Plot
      plotBackgroundColor = ChartsPanel.DEFAULT_PLOTBACKGROUNDCOLOR.getRGB();
      domainGridLineColor = ChartsPanel.DEFAULT_DOMAINGRIDLINECOLOR.getRGB();
      rangeGridLineColor = ChartsPanel.DEFAULT_RANGEGRIDLINECOLOR.getRGB();
      plotBaseShapeVisible = ChartsPanel.DEFAULT_PLOTBASESHAPEVISIBLE;
      plotBaseShapeFilled = ChartsPanel.DEFAULT_PLOTBASESHAPEVISIBLE;
      plotSeriesColors = new int[PLOT_SERIES_SEQUENCE_NUMBER];
      
      Paint[] chartSeriesColors = ChartSeriesColors.createDefaultPaintArray();
      
      for (int i = 0; i < PLOT_SERIES_SEQUENCE_NUMBER; i++)
         plotSeriesColors[i] = ((Color) chartSeriesColors[i]).getRGB();
       
      // Line Charts
      drawSeriesLinePath = ChartsPanel.DEFAULT_DRAWSERIESLINEPATH;
   }

   //==============================================================
   // Class methods to allow classes to get the default object
   // components.
   //==============================================================
   
   public int getTitleFontSize()
   {
      return titleFontSize;
   }
   
   public Color getChartForegroundColor()
   {
      return new Color(chartForegroundColor);
   }
   
   public Color getChartBackgroundColor()
   {
      return new Color(chartBackgroundColor);
   }
   
   public boolean getAutoSort()
   {
      return autoSort;
   }
   
   public boolean getAllowDuplicateXValues()
   {
      return allowDuplicateXValues;
   }
   
   public Color getPlotBackgroundColor()
   {
      return new Color(plotBackgroundColor);
   }
   
   public Color getDomainGridLineColor()
   {
      return new Color(domainGridLineColor);
   }
   
   public Color getRangeGridLineColor()
   {
      return new Color(rangeGridLineColor);
   }
   
   public boolean getPlotBaseShapeVisible()
   {
      return plotBaseShapeVisible;
   }
   
   public boolean getPlotBaseShapeFilled()
   {
      return plotBaseShapeFilled;
   }
   
   public Color getPlotSeriesColor(int key)
   {
      if (key < PLOT_SERIES_SEQUENCE_NUMBER)
         return new Color(plotSeriesColors[key]);
      else
         return Color.WHITE;
   }
   
   public boolean getDrawSeriesLinePath()
   {
      return drawSeriesLinePath;
   }
   
   //==============================================================
   // Class methods to allow classes to set the default object
   // components.
   //==============================================================
   
   public void setTitleFontSize(int value)
   {
      titleFontSize = value;
   }
   
   public void setChartForegroundColor(int value)
   {
      chartForegroundColor = value;
   }
   
   public void setChartBackgroundColor(int value)
   {
      chartBackgroundColor = value;
   }
   
   public void setAutoSort(boolean value)
   {
      autoSort = value;
   }
   
   public void setAllowDuplicateXValues(boolean value)
   {
      allowDuplicateXValues = value;
   }
   
   public void setPlotBackgroundColor(int value)
   {
      plotBackgroundColor = value;
   }
   
   public void setDomainGridLineColor(int value)
   {
      domainGridLineColor = value;
   }
   
   public void setRangeGridLineColor(int value)
   {
      rangeGridLineColor = value;
   }
   
   public void setPlotBaseShapeVisible(boolean value)
   {
      plotBaseShapeVisible = value;
   }
   
   public void setPlotBaseShapeFilled(boolean value)
   {
      plotBaseShapeFilled = value;
   }
   
   public void setPlotSeriesColor(int key, int value)
   {
      if (key < PLOT_SERIES_SEQUENCE_NUMBER)
      {
         plotSeriesColors[key] = value;
      }
   }
   
   public void setDrawSeriesLinePath(boolean value)
   {
      drawSeriesLinePath = value;
   }
   
   //==============================================================
   // Class method to properly implement the toString() method
   // for the object. Local method overides.
   //==============================================================

   public String toString()
   {
      StringBuffer parameters = new StringBuffer("[GeneralProperties: ");
      
      parameters.append("[titleFontSize = " + titleFontSize + "]");
      parameters.append("[chartForegroundColor = " + chartForegroundColor + "]");
      parameters.append("[chartBackgroundColor = " + chartBackgroundColor + "]");
      
      parameters.append("[autoSort = " + autoSort + "]");
      parameters.append("[allowDuplicateXValues = " + allowDuplicateXValues + "]");
      
      parameters.append("[plotBackgroundColor = " + plotBackgroundColor + "]");
      parameters.append("[domainGridLineColor = " + domainGridLineColor + "]");
      parameters.append("[rangeGridLineColor = " + rangeGridLineColor + "]");
      parameters.append("[plotBaseShapeVisible = " + plotBaseShapeVisible + "]");
      parameters.append("[plotBaseShapeFilled = " + plotBaseShapeFilled + "]");
      
      for (int i = 0; i < PLOT_SERIES_SEQUENCE_NUMBER; i++)
         parameters.append("[plotSeriesColors " + i + " = " + plotSeriesColors[i] + "]");
      
      parameters.append("[drawSeriesLinePath = " + drawSeriesLinePath + "]");
      
      return parameters.toString();
   }
}