//=================================================================
//                          ScatterPlot
//=================================================================
//
//    This class provides the main access point for setting up the
// requirements for the Java Scatter Plot plugin module for the
// lindyFrame framework.
//
//                      << ScatterPlot.java >>
//
//=================================================================
// Copyright (c) 2014-2020 Dana M. Proctor.
// Version 1.5 09/03/2020
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version
// 2 of the License, or (at your option) any later version. This
// program is distributed in the hope that it will be useful, 
// but WITHOUT ANY WARRANTY; without even the implied warranty
// of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See
// the GNU General Public License for more details. You should
// have received a copy of the GNU General Public License along
// with this program; if not, write to the Free Software Foundation,
// Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
// (http://opensource.org)
//
//=================================================================
// Revision History
// Changes to the code should be documented here and reflected
// in the present version number. Author information should
// also be included with the original copyright author.
//=================================================================
// Version 1.0 10/27/2017 Initial ScatterPlot Class.
//         1.1 11/16/2017 Removed Class Instance mainPanel & Returned Directly
//                        plotPanel in getPanel(). Changed CHART_DEFAULT_PROPERTIES
//                        to CHART_PROPERTIES. Cleaned Up.
//         1.2 11/18/2017 Removed this in Constructor From plotPanel Instantiation.
//                        Updated VERSION.
//         1.3 11/20/2017 Created Subset for CHART_OPTIONS & Included in Description
//                        Information for Librares.
//         1.4 12/09/2017 ScatterPlot Release v1.4. Updated VERSION. Synchronization
//                        With lindyFrame v2.9.
//         1.5 09/03/2020 Constructor Changed Resources to be Included in JAR.
//                        Updated VERSION. Synchronization with lindyFrame v4.1.
//                           
//-----------------------------------------------------------------
//                 danap@dandymadeproductions.com
//=================================================================

package com.dandymadeproductions.scatterplot;

import javax.swing.ImageIcon;
import javax.swing.JComponent;
import javax.swing.JMenuBar;
import javax.swing.JToolBar;

import com.dandymadeproductions.lindyframe.LindyFrame;
import com.dandymadeproductions.lindyframe.gui.Main_Frame;
import com.dandymadeproductions.lindyframe.utilities.LindyFrame_ResourceBundle;
import com.dandymadeproductions.scatterplot.charts.ChartProperties;

/**
 *    The ScatterPlot class provides the main access point for setting
 * up the requirements for the ScatterPlot plugin module for the Ajqvue
 * application.
 * 
 * @author Dana M. Proctor
 * @version 1.5 09/03/2020
 */

public class ScatterPlot
{
   // Class Instances
   private ImageIcon tabIcon;
   
   private PlotPanel plotPanel;

   private Plotter_MenuBar menuBar;
   private Plotter_ToolBar toolBar;

   private String imagesDirectory;
   private LindyFrame_ResourceBundle resourceBundle;
   
   public static final String LINE_CHART = "Line Chart";
   public static final String CATEGORY_LINE_CHART = "Category Line Chart";
   public static final String BAR_CHART = "Bar Chart";
   public static final String XYBAR_CHART = "XY Bar Chart";
   public static final String PIE_CHART = "Pie Chart";
   public static final String AREA_CHART = "Area Chart";
   public static final String XYAREA_CHART = "XY Area Chart";
   public static final String BUBBLE_CHART = "Bubble Chart";
   public static final String SCATTER_CHART = "Scatter Chart";

   //protected static final String[] CHART_OPTIONS = {LINE_CHART, CATEGORY_LINE_CHART, BAR_CHART,
   //                                                 XYBAR_CHART, PIE_CHART, AREA_CHART,
   //                                                 XYAREA_CHART, BUBBLE_CHART, SCATTER_CHART};
   
   protected static final String[] CHART_OPTIONS = {LINE_CHART, SCATTER_CHART};

   protected final static int FONT_SIZE = LindyFrame.getGeneralProperties().getFontSize();
   protected final static ChartProperties CHART_PROPERTIES = new ChartProperties();
   protected final static String DELIMITER = ",";

   private final static String VERSION = "Version 1.5";
   private final static String DESCRIPTION = "The ScatterPlot plugin provides a mean to demostrate"
                                             + " a plotting module example for the lindyFrame framework."
                                             + " The plugin is a basic Scatter Plot. ScatterPlot"
                                             + " includes the libraries (LGPL) JFreeChart and JCommon.";
    
   //==============================================================
   // ScatterPlot Constructor
   //==============================================================

   public ScatterPlot(Main_Frame parent, String path)
   {
      // Constructor Instances.
      MenuActionListener pluginMenuListener;
      String pathDirectory, localeDirectory;

      // Setup the Resources & Other Requirements.
      
      // ** NOTE: file/local network only, locale resource not in JAR **
      // pathDirectory = path + "/" + "ScatterPlot" + "/";
      // localeDirectory = "locale/";
      // imagesDirectory = "images/";
      // iconsDirectory = "lib/plugins/ScatterPlot/images/icons/";

      // ** NOTE: file/local network/http/ftp locale resource in JAR **
      pathDirectory = path + "/" + "ScatterPlot.jar";
      localeDirectory = "lib/plugins/ScatterPlot/locale/";
      imagesDirectory = "lib/plugins/ScatterPlot/images/";
      // iconsDirectory = "lib/plugins/ScatterPlot/images/icons/";

      resourceBundle = new LindyFrame_ResourceBundle(pathDirectory, LindyFrame.getDebug());
      resourceBundle.setLocaleResource(localeDirectory, "ScatterPlot", LindyFrame.getLocaleString());

      // Setup the Main pane and the plugin's components.

      tabIcon = resourceBundle.getResourceImage(imagesDirectory + "icons/chartsIcon.png");
      
      // Plot Panel.
      plotPanel = new PlotPanel(imagesDirectory, resourceBundle);

      // Setup the MenuBar and ToolBar to be used by the plugin.

      pluginMenuListener = new MenuActionListener(parent, plotPanel, resourceBundle, imagesDirectory);
      menuBar = new Plotter_MenuBar(parent, resourceBundle, pluginMenuListener);
      toolBar = new Plotter_ToolBar("ScatterPlot ToolBar", parent, imagesDirectory, resourceBundle,
                                    pluginMenuListener);
   }
   
 //==============================================================
   // Class method to return the plugin's JMenuBar
   //==============================================================

   protected JMenuBar getMenuBar()
   {
      return menuBar;
   }

   //==============================================================
   // Class method get the plugin's JToolBar
   //==============================================================

   protected JToolBar getToolBar()
   {
      return toolBar;
   }

   //==============================================================
   // Class method to get the main panel associated with the plugin.
   //==============================================================

   protected JComponent getPanel()
   {
      return plotPanel;
   }

   //==============================================================
   // Class method to get the plugin's version.
   //==============================================================

   protected String getVersion()
   {
      return VERSION;
   }

   //==============================================================
   // Class method to get the plugin's description.
   //==============================================================

   protected String getDescription()
   {
      return DESCRIPTION;
   }

   //==============================================================
   // Class method to get the icon that will be used in the
   // Ajqvue tab.
   //==============================================================

   protected ImageIcon getTabIcon()
   {
      return tabIcon;
   }
   
   //==============================================================
   // Class Method to return the current HeatMapperProperties.
   //==============================================================
   
   public static ChartProperties getChartDefaultProperties()
   {
      return CHART_PROPERTIES;
   }
}