//=================================================================
//                 ScatterPlot PreferencesPanel
//=================================================================
//
//    This class provides a panel that holds the components that
// are used to determine the ScatterPlot's preferences, properties,
// associated with running the module.
//
//                   << PreferencesPanel.java >>
//
//=================================================================
// Copyright (C) 2017 Dana M. Proctor.
// Version 1.1 11/19/2017
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version
// 2 of the License, or (at your option) any later version. This
// program is distributed in the hope that it will be useful, 
// but WITHOUT ANY WARRANTY; without even the implied warranty
// of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See
// the GNU General Public License for more details. You should
// have received a copy of the GNU General Public License along
// with this program; if not, write to the Free Software Foundation,
// Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
// (http://opensource.org)
//
//=================================================================
// Revision History
// Changes to the code should be documented here and reflected
// in the present version number. Author information should
// also be included with the original copyright author.
//=================================================================
// Version 1.0 11/01/2017 Original ScatterPlot PreferencesPanel Class.
//         1.1 11/19/2017 Replaced Method constraints() With Call to LindyFrame_Utils
//                        buildConstraints().
//
//-----------------------------------------------------------------
//                 danap@dandymadeproductions.com
//=================================================================

package com.dandymadeproductions.scatterplot;

import java.awt.BorderLayout;
import java.awt.Color;
import java.awt.GridBagConstraints;
import java.awt.GridBagLayout;
import java.awt.GridLayout;
import java.awt.Insets;
import java.awt.Paint;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;

import javax.swing.BorderFactory;
import javax.swing.BoxLayout;
import javax.swing.ImageIcon;
import javax.swing.JButton;
import javax.swing.JCheckBox;
import javax.swing.JColorChooser;
import javax.swing.JComponent;
import javax.swing.JDialog;
import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.JSpinner;
import javax.swing.SpinnerNumberModel;
import javax.swing.border.Border;
import javax.swing.border.EmptyBorder;

import com.dandymadeproductions.lindyframe.utilities.LindyFrame_ResourceBundle;
import com.dandymadeproductions.lindyframe.utilities.LindyFrame_Utils;
import com.dandymadeproductions.scatterplot.charts.ChartProperties;
import com.dandymadeproductions.scatterplot.charts.ChartsPanel;
import com.dandymadeproductions.scatterplot.data.ChartSeriesColors;

/**
 *    The PreferencesPanel class provides a panel that holds the
 * components that are used to determine the ScatterPlot preferences,
 * properties, associated with running the module.
 *    
 * @author Dana M. Proctor
 * @version 1.1 11/19/2017
 */

public class PreferencesPanel extends JPanel implements ActionListener
{
   // Class Instances
   private static final long serialVersionUID = 7038356098876284781L;
   
   private ChartProperties chartProperties;
   private GridBagConstraints constraints;
   private GridBagLayout gridbag;
   private JColorChooser panelColorChooser;
   private String actionCommand;
   private ImageIcon transparentIcon;
   
   private JSpinner titleFontSizeSpinner;
   private JButton chartBackgndColorButton;
   private JButton chartForegndColorButton;
   
   private JCheckBox autoSortCheckBox;
   private JCheckBox allowDuplicateXValuesCheckBox;
   
   private JButton plotBackgndColorButton;
   private JButton domainGridLineColorButton;
   private JButton rangeGridLineColorButton;
   private JCheckBox plotBaseShapeVisibleCheckBox;
   private JCheckBox plotBaseShapeFilledCheckBox;
   private JButton[] plotSeriesColorsButtons;
   
   private JCheckBox drawSeriesLinePathCheckBox;
   
   private JButton restoreDefaultsButton;
   
   public static final int MINIMUM_FONT_SIZE = 8;
   public static final int MAX_FONT_SIZE = 32;
   public static final int SPINNER_SIZE_STEP = 1;
   
   private static final String BACKGROUND = "background";
   private static final String FOREGROUND = "foreground";
   private static final String PLOT = "plot";
   private static final String PLOT_SERIES = "plot series";
   private static final String DOMAIN = "domain";
   private static final String RANGE = "range";
   
   //==============================================================
   // PreferencesPanel Constructor
   //==============================================================
   
   public PreferencesPanel(LindyFrame_ResourceBundle resourceBundle, String imagesDirectory)
   {  
      // Constructor Instances
      JPanel mainPanel, southButtonPanel;
      
      String resource;
      Border outerBorder;
      SpinnerNumberModel titleFontSizeSpinnerModel;
      
      // Setup panel and resource source.
      setLayout(new BorderLayout());
      
      outerBorder = BorderFactory.createCompoundBorder(
         BorderFactory.createEtchedBorder(), BorderFactory.createLoweredBevelBorder());
      setBorder(BorderFactory.createCompoundBorder(outerBorder,
                                                   BorderFactory.createEmptyBorder(5, 0, 0, 0)));
      
      chartProperties = ScatterPlot.getChartDefaultProperties();
      gridbag = new GridBagLayout();
      constraints = new GridBagConstraints();
      panelColorChooser = LindyFrame_Utils.createColorChooser(this);
      transparentIcon = resourceBundle.getResourceImage(imagesDirectory + "icons/transparentUpIcon.png");
      
      // Main components panel container.
      mainPanel = new JPanel();
      mainPanel.setLayout(new BoxLayout(mainPanel, BoxLayout.Y_AXIS));
      mainPanel.setBorder(new EmptyBorder(2, 5, 2, 5));
      
      // Preferences Options Components.
      
      //===========
      // Chart
      
      JPanel chartPanel = new JPanel(new GridLayout(3, 1, 0, 4));
      chartPanel.setBorder(BorderFactory.createEmptyBorder(2, 0, 10, 0));
      
      // Title Font Size
      JPanel titleFontSizePanel = new JPanel(gridbag);
      
      titleFontSizeSpinnerModel = new SpinnerNumberModel(chartProperties.getTitleFontSize(),
                                                         MINIMUM_FONT_SIZE, MAX_FONT_SIZE,
                                                         SPINNER_SIZE_STEP);
      titleFontSizeSpinner = new JSpinner(titleFontSizeSpinnerModel);
      
      resource = resourceBundle.getResourceString("PreferencesPanel.label.TitleFontSize", "Title Font Size");
      JLabel titleFontSizeLabel = new JLabel(resource);
      
      setupPanel(titleFontSizeSpinner, titleFontSizeLabel, titleFontSizePanel);
      chartPanel.add(titleFontSizePanel);
      
      // Chart Background Color
      JPanel chartBackgndColorPanel = new JPanel(gridbag);
      
      chartBackgndColorButton = new JButton(transparentIcon);
      chartBackgndColorButton.setBackground(chartProperties.getChartBackgroundColor());
      chartBackgndColorButton.setFocusable(false);
      chartBackgndColorButton.setMargin(new Insets(0, 0, 0, 0));
      chartBackgndColorButton.setActionCommand(BACKGROUND);
      chartBackgndColorButton.addActionListener(this);
      
      resource = resourceBundle.getResourceString("PreferencesPanel.label.ChartBackgroundColor",
                                                  "Chart Background Color");
      JLabel chartBackgndColorLabel = new JLabel(resource);
      chartBackgndColorLabel.setBorder(BorderFactory.createEmptyBorder(0, 6, 0, 0));
      
      setupPanel(chartBackgndColorButton, chartBackgndColorLabel, chartBackgndColorPanel);
      chartPanel.add(chartBackgndColorPanel);
      
      // Chart foreground Color
      JPanel chartForegndColorPanel = new JPanel(gridbag);
      
      chartForegndColorButton = new JButton(transparentIcon);
      chartForegndColorButton.setBackground(chartProperties.getChartForegroundColor());
      chartForegndColorButton.setFocusable(false);
      chartForegndColorButton.setMargin(new Insets(0, 0, 0, 0));
      chartForegndColorButton.setActionCommand(FOREGROUND);
      chartForegndColorButton.addActionListener(this);
      
      resource = resourceBundle.getResourceString("PreferencesPanel.label.ChartForegroundColor",
                                                  "Chart Foreground Color");
      JLabel chartForegndColorLabel = new JLabel(resource);
      chartForegndColorLabel.setBorder(BorderFactory.createEmptyBorder(0, 6, 0, 0));
      
      setupPanel(chartForegndColorButton, chartForegndColorLabel, chartForegndColorPanel);
      chartPanel.add(chartForegndColorPanel);
      
      mainPanel.add(chartPanel);
      
      //===========
      // Series
      
      JPanel seriesPanel = new JPanel(new GridLayout(2, 1, 0, 6));
      seriesPanel.setBorder(BorderFactory.createEmptyBorder(2, 0, 10, 0));
      
      // Auto Sort
      resource = resourceBundle.getResourceString("PreferencesPanel.label.AutoSortX-Axis",
                                                  "Auto Sort X-Axis");
      autoSortCheckBox = new JCheckBox(resource, chartProperties.getAutoSort());
      autoSortCheckBox.setBorder(BorderFactory.createEmptyBorder(1, 2, 1, 0));
      seriesPanel.add(autoSortCheckBox);
      
      // Auto Duplicate X Values
      resource = resourceBundle.getResourceString("PreferencesPanel.label.AllowDuplicateXValues",
                                                  "Allow Duplicate X Values");
      allowDuplicateXValuesCheckBox = new JCheckBox(resource, chartProperties.getAllowDuplicateXValues());
      allowDuplicateXValuesCheckBox.setBorder(BorderFactory.createEmptyBorder(1, 2, 1, 0));
      seriesPanel.add(allowDuplicateXValuesCheckBox);
      
      mainPanel.add(seriesPanel);
      
      //===========
      // Plot
      
      JPanel plotPanel = new JPanel(new GridLayout(6, 1, 0, 4));
      plotPanel.setBorder(BorderFactory.createEmptyBorder(2, 0, 10, 0));
      
      // Plot Background Color
      JPanel plotBackgndColorPanel = new JPanel(gridbag);
      
      plotBackgndColorButton = new JButton(transparentIcon);
      plotBackgndColorButton.setBackground(chartProperties.getPlotBackgroundColor());
      plotBackgndColorButton.setFocusable(false);
      plotBackgndColorButton.setMargin(new Insets(0, 0, 0, 0));
      plotBackgndColorButton.setActionCommand(PLOT);
      plotBackgndColorButton.addActionListener(this);
      
      resource = resourceBundle.getResourceString("PreferencesPanel.label.PlotBackgroundColor",
                                                  "Plot Background Color");
      JLabel plotBackgndColorLabel = new JLabel(resource);
      plotBackgndColorLabel.setBorder(BorderFactory.createEmptyBorder(0, 6, 0, 0));
      
      setupPanel(plotBackgndColorButton, plotBackgndColorLabel, plotBackgndColorPanel);
      plotPanel.add(plotBackgndColorPanel);
      
      // Domain Gridline Color
      JPanel domainGridlineColorPanel = new JPanel(gridbag);
      
      domainGridLineColorButton = new JButton(transparentIcon);
      domainGridLineColorButton.setBackground(chartProperties.getDomainGridLineColor());
      domainGridLineColorButton.setFocusable(false);
      domainGridLineColorButton.setMargin(new Insets(0, 0, 0, 0));
      domainGridLineColorButton.setActionCommand(DOMAIN);
      domainGridLineColorButton.addActionListener(this);
      
      resource = resourceBundle.getResourceString("PreferencesPanel.label.DomainGridLineColor",
                                                  "Domain Grid Line Color");
      JLabel domainGridLineColorLabel = new JLabel(resource);
      domainGridLineColorLabel.setBorder(BorderFactory.createEmptyBorder(0, 6, 0, 0));
      
      setupPanel(domainGridLineColorButton, domainGridLineColorLabel, domainGridlineColorPanel);
      plotPanel.add(domainGridlineColorPanel);
      
      // Range Gridline Color
      JPanel rangeGridlineColorPanel = new JPanel(gridbag);
      
      rangeGridLineColorButton = new JButton(transparentIcon);
      rangeGridLineColorButton.setBackground(chartProperties.getRangeGridLineColor());
      rangeGridLineColorButton.setFocusable(false);
      rangeGridLineColorButton.setMargin(new Insets(0, 0, 0, 0));
      rangeGridLineColorButton.setActionCommand(RANGE);
      rangeGridLineColorButton.addActionListener(this);
      
      resource = resourceBundle.getResourceString("PreferencesPanel.label.RangeGridLineColor",
                                                  "Range Grid Line Color");
      JLabel rangeGridLineColorLabel = new JLabel(resource);
      rangeGridLineColorLabel.setBorder(BorderFactory.createEmptyBorder(0, 6, 0, 0));
      
      setupPanel(rangeGridLineColorButton, rangeGridLineColorLabel, rangeGridlineColorPanel);
      plotPanel.add(rangeGridlineColorPanel);
      
      // Plot Base Shape Visible
      resource = resourceBundle.getResourceString("PreferencesPanel.label.PlotShapeVisible",
                                                  "Plot Shape Visible");
      plotBaseShapeVisibleCheckBox = new JCheckBox(resource, chartProperties.getPlotBaseShapeVisible());
      plotBaseShapeVisibleCheckBox.setBorder(BorderFactory.createEmptyBorder(1, 2, 1, 0));
      plotPanel.add(plotBaseShapeVisibleCheckBox);
      
      // Plot Base Shape Filled
      resource = resourceBundle.getResourceString("PreferencesPanel.label.PlotShapeFilled",
                                                  "Plot Shape Filled");
      plotBaseShapeFilledCheckBox = new JCheckBox(resource, chartProperties.getPlotBaseShapeFilled());
      plotBaseShapeFilledCheckBox.setBorder(BorderFactory.createEmptyBorder(1, 2, 1, 0));
      plotPanel.add(plotBaseShapeFilledCheckBox);
      
      // Draw Series Line Path
      resource = resourceBundle.getResourceString("PreferencesPanel.label.DrawLinePath",
                                                  "DrawLinePath");
      drawSeriesLinePathCheckBox = new JCheckBox(resource, chartProperties.getDrawSeriesLinePath());
      drawSeriesLinePathCheckBox.setBorder(BorderFactory.createEmptyBorder(1, 2, 1, 0));
      plotPanel.add(drawSeriesLinePathCheckBox);
      
      mainPanel.add(plotPanel);
      
      // Plot Paint Series
      mainPanel.add(setupPaintSeriesButtons(resourceBundle));
      
      add(mainPanel, BorderLayout.CENTER);

      // South panel to contain the defaults & apply
      // buttons.
      
      southButtonPanel = new JPanel();
      southButtonPanel.setBorder(BorderFactory.createRaisedBevelBorder());

      // Restore Defaults
      resource = resourceBundle.getResourceString("PreferencesPanel.button.RestoreDefaults",
                                                  "Restore Defaults");
      restoreDefaultsButton = new JButton(resource);
      restoreDefaultsButton.setFocusPainted(false);
      restoreDefaultsButton.addActionListener(this);
      southButtonPanel.add(restoreDefaultsButton);
      
      add(southButtonPanel, BorderLayout.SOUTH);
   }
   
   //==============================================================
   // ActionEvent Listener method for determining when the restore
   // defaults properties button is activiated.
   //==============================================================

   public void actionPerformed(ActionEvent evt)
   {
      Object panelSource = evt.getSource();

      if (panelSource instanceof JButton)
      {
         // Restore Defaults Action
         if (panelSource == restoreDefaultsButton)
         {
            titleFontSizeSpinner.setValue(ScatterPlot.FONT_SIZE + 2); 
            chartBackgndColorButton.setBackground(ChartsPanel.DEFAULT_BACKGROUND);
            chartForegndColorButton.setBackground(ChartsPanel.DEFAULT_FOREGROUND);
            
            autoSortCheckBox.setSelected(ChartsPanel.DEFAULT_AUTOSORT);
            allowDuplicateXValuesCheckBox.setSelected(ChartsPanel.DEFAULT_ALLOWDUPLICATEXVALUES);
            
            plotBackgndColorButton.setBackground(ChartsPanel.DEFAULT_PLOTBACKGROUNDCOLOR);
            domainGridLineColorButton.setBackground(ChartsPanel.DEFAULT_DOMAINGRIDLINECOLOR);
            rangeGridLineColorButton.setBackground(ChartsPanel.DEFAULT_RANGEGRIDLINECOLOR);
            plotBaseShapeVisibleCheckBox.setSelected(ChartsPanel.DEFAULT_PLOTBASESHAPEVISIBLE);
            plotBaseShapeFilledCheckBox.setSelected(ChartsPanel.DEFAULT_PLOTBASESHAPEFILLED);
            
            Paint[] defaultPaintSeries = ChartSeriesColors.createDefaultPaintArray();
            
            for (int i = 0; i < ChartProperties.PLOT_SERIES_SEQUENCE_NUMBER; i++)
               plotSeriesColorsButtons[i].setBackground(((Color) defaultPaintSeries[i]));
            
            drawSeriesLinePathCheckBox.setSelected(ChartsPanel.DEFAULT_DRAWSERIESLINEPATH);
         }
         else
         {
            // Title
            if (panelSource == chartBackgndColorButton || panelSource == chartForegndColorButton
                || panelSource == plotBackgndColorButton || panelSource == domainGridLineColorButton
                || panelSource == rangeGridLineColorButton || isPlotSeriesColorsButton(panelSource))
            {
               JButton currentButton = (JButton) panelSource;
               actionCommand = currentButton.getActionCommand();

               if (currentButton == chartBackgndColorButton)
                  panelColorChooser.setColor(chartBackgndColorButton.getBackground());
               else if (currentButton == chartForegndColorButton)
                  panelColorChooser.setColor(chartForegndColorButton.getBackground());
               else if (currentButton == plotBackgndColorButton)
                  panelColorChooser.setColor(plotBackgndColorButton.getBackground());
               else if (currentButton == domainGridLineColorButton)
                  panelColorChooser.setColor(domainGridLineColorButton.getBackground());
               else if (currentButton == rangeGridLineColorButton)
                  panelColorChooser.setColor(rangeGridLineColorButton.getBackground());
               else
               {
                  for (int i = 0; i < ChartProperties.PLOT_SERIES_SEQUENCE_NUMBER; i++)
                     if (currentButton == plotSeriesColorsButtons[i])
                        panelColorChooser.setColor(plotSeriesColorsButtons[i].getBackground());
               }

               // Create the color chooser dialog.
               JDialog dialog;
               dialog = JColorChooser.createDialog(this, "Color Selector", true, panelColorChooser, this, null);
               dialog.setVisible(true);
               dialog.dispose();
               return;
            }

            // Color Chooser Action
            if (((JButton) panelSource).getText().equals("OK"))
            {  
               if (actionCommand.equals(BACKGROUND))
                  chartBackgndColorButton.setBackground(panelColorChooser.getColor());
               else if (actionCommand.equals(FOREGROUND))
                  chartForegndColorButton.setBackground(panelColorChooser.getColor());
               else if (actionCommand.equals(PLOT))
                  plotBackgndColorButton.setBackground(panelColorChooser.getColor());
               else if (actionCommand.equals(DOMAIN))
                  domainGridLineColorButton.setBackground(panelColorChooser.getColor());
               else if (actionCommand.equals(RANGE))
                  rangeGridLineColorButton.setBackground(panelColorChooser.getColor());
               else
               {
                  for (int i = 0; i < ChartProperties.PLOT_SERIES_SEQUENCE_NUMBER; i++)
                     if (actionCommand.equals(PLOT_SERIES + i))
                        plotSeriesColorsButtons[i].setBackground(panelColorChooser.getColor());
               }

               actionCommand = "";
            }
         }    
      }
   }
   
   //================================================================
   // Class Method to help determine pane source of plot series
   // colors buttons.
   //================================================================

   private boolean isPlotSeriesColorsButton(Object panelSource)
   {
      // Method Instances
      boolean isPlotSeriesButton = false;
      
      for (int i = 0; i < ChartProperties.PLOT_SERIES_SEQUENCE_NUMBER; i++)
         if (panelSource == plotSeriesColorsButtons[i])
            isPlotSeriesButton = true;
      
      return isPlotSeriesButton;
   }
   
   //================================================================
   // Class Method for helping setting up two component panels.
   //================================================================

   private void setupPanel(JComponent lftComp, JComponent rhtComp, JPanel panel)
   {
      LindyFrame_Utils.buildConstraints(constraints, 0, 0, 1, 1, 15, 100);
      constraints.fill = GridBagConstraints.NONE;
      constraints.anchor = GridBagConstraints.WEST;
      gridbag.setConstraints(lftComp, constraints);
      panel.add(lftComp);
      
      LindyFrame_Utils.buildConstraints(constraints, 1, 0, 1, 1, 85, 100);
      constraints.fill = GridBagConstraints.NONE;
      constraints.anchor = GridBagConstraints.WEST;
      gridbag.setConstraints(rhtComp, constraints);
      panel.add(rhtComp);
   }
   
   //================================================================
   // Class Method for helping setting up the paint series buttons.
   //================================================================

   private JPanel setupPaintSeriesButtons(LindyFrame_ResourceBundle resourceBundle)
   {
      // Method Instances
      JPanel seriesButtonPanel;
      String resource;
      
      // Create series button group.
      seriesButtonPanel = new JPanel(gridbag);
      seriesButtonPanel.setBorder(BorderFactory.createEmptyBorder(4, 0, 4, 0));
      
      resource = resourceBundle.getResourceString("PreferencesPanel.label.PlotSeriesColors",
                                                  "Plot Series Colors");
      JLabel plotSeriesColorsLabel = new JLabel(resource);
      
      LindyFrame_Utils.buildConstraints(constraints, 0, 0, ChartProperties.PLOT_SERIES_SEQUENCE_NUMBER, 1, 100, 50);
      constraints.fill = GridBagConstraints.NONE;
      constraints.anchor = GridBagConstraints.CENTER;
      gridbag.setConstraints(plotSeriesColorsLabel, constraints);
      seriesButtonPanel.add(plotSeriesColorsLabel);
      
      plotSeriesColorsButtons = new JButton[ChartProperties.PLOT_SERIES_SEQUENCE_NUMBER];
      
      for (int i = 0; i < ChartProperties.PLOT_SERIES_SEQUENCE_NUMBER; i++)
      {
         plotSeriesColorsButtons[i] = new JButton(transparentIcon);
         plotSeriesColorsButtons[i].setBackground(chartProperties.getPlotSeriesColor(i));
         plotSeriesColorsButtons[i].setFocusable(false);
         plotSeriesColorsButtons[i].setMargin(new Insets(0, 0, 0, 0));
         plotSeriesColorsButtons[i].setActionCommand(PLOT_SERIES + i);
         plotSeriesColorsButtons[i].addActionListener(this);
         
         LindyFrame_Utils.buildConstraints(constraints, i, 1, 1, 1, 100, (i == 0 ? 50:0));
         constraints.fill = GridBagConstraints.NONE;
         constraints.anchor = GridBagConstraints.CENTER;
         gridbag.setConstraints(plotSeriesColorsButtons[i], constraints);
         seriesButtonPanel.add(plotSeriesColorsButtons[i]);
      }
      return seriesButtonPanel;
   }
   
   //==============================================================
   // Class Method to derives the options set in the panel and
   // apply them to the QueryBuilder properties settings.
   //==============================================================
   
   public void setProperties()
   {
      ChartProperties chartProperties = ScatterPlot.getChartDefaultProperties();
      
      chartProperties.setTitleFontSize((int) titleFontSizeSpinner.getValue()); 
      chartProperties.setChartBackgroundColor(chartBackgndColorButton.getBackground().getRGB());
      chartProperties.setChartForegroundColor(chartForegndColorButton.getBackground().getRGB());
      
      chartProperties.setAutoSort(autoSortCheckBox.isSelected());
      chartProperties.setAllowDuplicateXValues(allowDuplicateXValuesCheckBox.isSelected());
      
      chartProperties.setPlotBackgroundColor(plotBackgndColorButton.getBackground().getRGB());
      chartProperties.setDomainGridLineColor(domainGridLineColorButton.getBackground().getRGB());
      chartProperties.setRangeGridLineColor(rangeGridLineColorButton.getBackground().getRGB());
      chartProperties.setPlotBaseShapeVisible(plotBaseShapeVisibleCheckBox.isSelected());
      chartProperties.setPlotBaseShapeFilled(plotBaseShapeFilledCheckBox.isSelected());
      
      for (int i = 0; i < ChartProperties.PLOT_SERIES_SEQUENCE_NUMBER; i++)
         chartProperties.setPlotSeriesColor(i, plotSeriesColorsButtons[i].getBackground().getRGB());
      
      chartProperties.setDrawSeriesLinePath(drawSeriesLinePathCheckBox.isSelected());
   }
}