//=================================================================
//                        PlotPanel
//=================================================================
//
//    This class provides the panel that holds all the component
// used in the ScatterPlot.
//
//                   << PlotPanel.java >>
//
//=================================================================
// Copyright (C) 2017 Dana M. Proctor
// Version 1.1 11/20/2017
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version
// 2 of the License, or (at your option) any later version. This
// program is distributed in the hope that it will be useful, 
// but WITHOUT ANY WARRANTY; without even the implied warranty
// of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See
// the GNU General Public License for more details. You should
// have received a copy of the GNU General Public License along
// with this program; if not, write to the Free Software Foundation,
// Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
// (http://opensource.org)
//
//=================================================================
// Revision History
// Changes to the code should be documented here and reflected
// in the present version number. Author information should
// also be included with the original copyright author.
//=================================================================
// Version 1.0 10/30/2017 Initial PlotPanel Class.
//         1.1 11/20/2017 Constructor Changed chartSelectorComboBox to Include
//                        Line Chart, via ScatterPlot.CHART_OPTIONS. Method
//                        setChartType() Included Line Chart & Corrected Selectors
//                        for Possible Additional Charts.
//                         
//-----------------------------------------------------------------
//                 danap@dandymadeproductions.com
//=================================================================

package com.dandymadeproductions.scatterplot;

import java.awt.BorderLayout;
import java.awt.GridBagConstraints;
import java.awt.GridBagLayout;
import java.awt.GridLayout;
import java.awt.Insets;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;

import javax.swing.BorderFactory;
import javax.swing.ImageIcon;
import javax.swing.JButton;
import javax.swing.JComboBox;
import javax.swing.JPanel;

import com.dandymadeproductions.lindyframe.utilities.LindyFrame_ResourceBundle;
import com.dandymadeproductions.lindyframe.utilities.LindyFrame_Utils;
import com.dandymadeproductions.scatterplot.charts.ChartsPanel;
import com.dandymadeproductions.scatterplot.charts.LineChartPanel;
import com.dandymadeproductions.scatterplot.charts.ScatterChartPanel;

/**
 *    The PlotPanel class provides the panel that holds all the
 * components used in the ScatterPlot plugin for showing a selected
 * chart.  
 * 
 * @author Dana M. Proctor
 * @version 1.1 11/20/2017
 */

public class PlotPanel extends JPanel implements ActionListener
{
   private static final long serialVersionUID = 117774406001854773L;
   
   // Class Instances
   private JPanel dataPanel;
   private ChartsPanel chartsPanel;
   
   private JComboBox<String> chartSelectorComboBox;
   private JButton refreshButton;
   private JButton clearButton;
   
   private boolean disableActions;
   
   private String iconsDirectory;
   private String resourceExample, resourceChart;
   
   //==============================================================
   // PlotPanel Constructor
   //==============================================================

   protected PlotPanel(String imagesDirectory, LindyFrame_ResourceBundle resourceBundle)
   {  
      // Constructor Instances
      JPanel controlPanel;
      JPanel centerCtlPanel, rightCtlPanel;

      ImageIcon refreshButtonIcon, clearButtonIcon;
      
      GridBagLayout gridbag;
      GridBagConstraints constraints;
      
      // Setup
      
      disableActions = false;
      
      resourceExample = resourceBundle.getResourceString("PlotPanel.label.Example", "Example");
      resourceChart = resourceBundle.getResourceString("PlotPanel.label.Chart", "Chart");
      
      iconsDirectory = imagesDirectory + "icons/";
      
      setLayout(new BorderLayout());

      gridbag = new GridBagLayout();
      constraints = new GridBagConstraints();

      // ===============================================
      // Create the control panel.

      controlPanel = new JPanel(new GridLayout(1, 1, 0, 0));

      // Create SQL query access and chart selector.

      centerCtlPanel = new JPanel(gridbag);
      centerCtlPanel.setBorder(BorderFactory.createCompoundBorder(BorderFactory
            .createLoweredBevelBorder(), BorderFactory.createEmptyBorder(0, 0, 0, 0)));
      
      // Chart Selector
      chartSelectorComboBox = new JComboBox<String>(ScatterPlot.CHART_OPTIONS);
      chartSelectorComboBox.setBorder(BorderFactory.createRaisedBevelBorder());
      chartSelectorComboBox.setSelectedItem(ScatterPlot.SCATTER_CHART);
      chartSelectorComboBox.addActionListener(this);
      
      LindyFrame_Utils.buildConstraints(constraints, 1, 0, 1, 1, 96, 100);
      constraints.fill = GridBagConstraints.HORIZONTAL;
      constraints.anchor = GridBagConstraints.WEST;
      gridbag.setConstraints(chartSelectorComboBox, constraints);
      centerCtlPanel.add(chartSelectorComboBox);
      
      LindyFrame_Utils.buildConstraints(constraints, 0, 0, 1, 1, 90, 100);
      constraints.fill = GridBagConstraints.NONE;
      constraints.anchor = GridBagConstraints.CENTER;
      gridbag.setConstraints(centerCtlPanel, constraints);
      controlPanel.add(centerCtlPanel);
      
      // Create control buttons, refresh, timer, & clear.
      
      rightCtlPanel = new JPanel(gridbag);
      rightCtlPanel.setBorder(BorderFactory.createCompoundBorder(
         BorderFactory.createLoweredBevelBorder(), BorderFactory.createEmptyBorder(0, 0, 0, 0)));
      
      // Refresh Chart
      refreshButtonIcon = resourceBundle.getResourceImage(iconsDirectory + "refreshIcon.png");
      
      refreshButton = new JButton(refreshButtonIcon);
      refreshButton.setMargin(new Insets(0, 0, 0, 0));
      refreshButton.setBorder(BorderFactory.createCompoundBorder(
         BorderFactory.createRaisedBevelBorder(), refreshButton.getBorder()));
      refreshButton.setFocusPainted(false);
      refreshButton.addActionListener(this); 
      
      LindyFrame_Utils.buildConstraints(constraints, 0, 0, 1, 1, 50, 100);
      constraints.fill = GridBagConstraints.HORIZONTAL;
      constraints.anchor = GridBagConstraints.CENTER;
      gridbag.setConstraints(refreshButton, constraints);
      rightCtlPanel.add(refreshButton);
      
      // Clear Chart/Query
      clearButtonIcon = resourceBundle.getResourceImage(iconsDirectory + "clearIcon.png");

      clearButton = new JButton(clearButtonIcon);
      clearButton.setMargin(new Insets(0, 0, 0, 0));
      clearButton.setBorder(BorderFactory.createCompoundBorder(
         BorderFactory.createRaisedBevelBorder(), clearButton.getBorder()));
      clearButton.setFocusPainted(false);
      clearButton.addActionListener(this); 
      
      LindyFrame_Utils.buildConstraints(constraints, 2, 0, 1, 1, 50, 100);
      constraints.fill = GridBagConstraints.HORIZONTAL;
      constraints.anchor = GridBagConstraints.CENTER;
      gridbag.setConstraints(clearButton, constraints);
      rightCtlPanel.add(clearButton);

      LindyFrame_Utils.buildConstraints(constraints, 1, 0, 1, 1, 10, 100);
      constraints.fill = GridBagConstraints.NONE;
      constraints.anchor = GridBagConstraints.CENTER;
      gridbag.setConstraints(rightCtlPanel, constraints);
      controlPanel.add(rightCtlPanel);

      add(controlPanel, BorderLayout.NORTH);

      // ===============================================
      // Create the chart panel.
      
      dataPanel = new JPanel(new GridLayout(1, 1, 0, 0));
      dataPanel.setBorder(BorderFactory.createCompoundBorder(BorderFactory.createEtchedBorder(),
         BorderFactory.createLoweredBevelBorder()));
      
      chartsPanel = new ScatterChartPanel(resourceExample, resourceChart);
      
      dataPanel.add(chartsPanel);
      add(dataPanel, BorderLayout.CENTER);
   }

   //==============================================================
   // ActionEvent Listener method for detecting the inputs from the
   // panel and directing to the appropriate routine.
   //==============================================================

   public void actionPerformed(ActionEvent evt)
   {
      Object panelSource = evt.getSource();

      if (disableActions)
         return;

      // Chart Selector Action
      if (panelSource instanceof JComboBox)
      {
         if (panelSource == chartSelectorComboBox)
         {
            setChartType(chartSelectorComboBox.getSelectedItem()); 
         }
      }

      // Query Form, & Query Form Close Actions
      if (panelSource instanceof JButton)
      {
         // Refresh Action
         if (panelSource == refreshButton)
         {
            // Set Example Data.
            chartsPanel.setExampleData();
            return;
         }
         
         // Clear Action
         if (panelSource == clearButton)
         {
            chartsPanel.removeAllData();
         } 
      }
   }
   
   //==============================================================
   // Method for setting the user's choice for type of chart to be
   // used.
   //==============================================================
   
   private void setChartType(Object chartType)
   {  
      dataPanel.removeAll();
      
      
      if (chartType.equals(ScatterPlot.LINE_CHART))
         chartsPanel = new LineChartPanel(resourceExample, resourceChart);
      /*
      else if (chartType.equals(ScatterPlot.CATEGORY_LINE_CHART))
         chartsPanel = new CategoryLineChartPanel(resourceExample, resourceChart);
      else if (chartType.equals(ScatterPlot.BAR_CHART))
         chartsPanel = new BarChartPanel(resourceExample, resourceChart);
      else if (chartType.equals(ScatterPlot.XYBAR_CHART))
         chartsPanel = new XYBarChartPanel(resourceExample, resourceChart);
      else if (chartType.equals(ScatterPlot.PIE_CHART))
         chartsPanel = new PieChartPanel(resourceExample, resourceChart);
      else if (chartType.equals(ScatterPlot.AREA_CHART))
         chartsPanel = new AreaChartPanel(resourceExample, resourceChart);
      else if (chartType.equals(ScatterPlot.XYAREA_CHART))
         chartsPanel = new XYAreaChartPanel(resourceExample, resourceChart);
      else if (chartType.equals(ScatterPlot.BUBBLE_CHART))
         chartsPanel = new BubbleChartPanel(resourceExample, resourceChart);
      */
      else if (chartType.equals(ScatterPlot.SCATTER_CHART))
         chartsPanel = new ScatterChartPanel(resourceExample, resourceChart);
      else
         chartsPanel = new ChartsPanel();
      
      dataPanel.add(chartsPanel);
      validate();
   }
   
   //==============================================================
   // Method to set the disable actions instance.
   //==============================================================
   
   protected void setDisableActions(boolean value)
   {
      disableActions = value;
      chartSelectorComboBox.setEnabled(!disableActions); 
   }
   
   //==============================================================
   // Class method to return the main pane holding the chart.
   //==============================================================
   
   public ChartsPanel getChartsPanel()
   {
      return chartsPanel;
   }
   
   //==============================================================
   // Class method to return the main pane holding the chart.
   //==============================================================
   
   public String getChartType()
   {
      return (String) chartSelectorComboBox.getSelectedItem();
   }
}