//=================================================================
//                        HTMLEditorPane
//=================================================================
//
//    This class is used to view HTML based documentations. The
// subclass implements search and highlight functionality using a
// base search mathoc by calling String.indexOf() for the matching
// word.
//
//                   << HTMLEditorPane.java >>
//
//=================================================================
// Copyright (C) 2005-2017 Dana M. Proctor
// Version 1.2 04/26/2017
//
// This file is part of JApiViewer.
//
// JApiViewer is free software; you can redistribute it and/or modify it under
// the terms of the GNU General Public License as published by the Free Software
// Foundation; either version 2 of the License, or (at your option) any later
// version.
// 
// JApiViewer is distributed in the hope that it will be useful, but WITHOUT ANY
// WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
// A PARTICULAR PURPOSE. See the GNU General Public License for more details.
// 
// You should have received a copy of the GNU General Public License along with
// JApiViewer; if not, write to the Free Software Foundation, Inc., 51 Franklin
// St, Fifth Floor, Boston, MA 02110-1301 USA
//
//=================================================================
// Revision History
// Changes to the code should be documented here and reflected
// in the present version number. Author information should
// also be included with the original copyright author.
//=================================================================
// Version 1.0 2005 Initial JApiViewer ApiHTMLViewer Class.
//         1.1 08/02/2013 Modified to be Used With HTMLViewer Plugin
//                        for lindyFrame Framework. Renamed HTMLEditorPane.
//         1.2 04/26/2017 Updated Copyright, Added Version in javadoc Constructor
//                        Comment & Corrected Parameter for Class SearchWord
//                        Method search() By Providing Description, Jre8 javadoc
//                        Error.
//                           
//-----------------------------------------------------------------
//                 Thasso@randgestalten.de
//                 danap@dandymadeproductions.com
//=================================================================

package com.dandymadeproductions.htmlviewer;

import java.awt.Color;
import java.io.IOException;
import java.net.URL;

import javax.swing.JEditorPane;
import javax.swing.text.BadLocationException;
import javax.swing.text.DefaultHighlighter;
import javax.swing.text.Document;
import javax.swing.text.Highlighter;
import javax.swing.text.JTextComponent;
import javax.swing.text.Highlighter.Highlight;

/**
 *  The HTMLEditorPane class is used to view HTML based documentations.
 * This subclass implements search and highlight functionality using
 * a base search mathoc by calling String.indexOf() for the matching
 * word.
 * 
 * @author Thasso
 * @author Dana M. Proctor
 * @version 1.2 04/26/2017
 */

public class HTMLEditorPane extends JEditorPane
{
   private static final long serialVersionUID = -1388878534761954412L;
   
   /**
    * the searcher
    */
   protected WordSearcher searcher = new WordSearcher(this);
   /**
    * last search word
    */
   protected String lastSearch = null;
   /**
    * save the highlights here
    */
   protected Highlight h[];
   /**
    * The actual selected highlight index
    */
   protected int index = 0;

   /**
    * Sets the page of the JEditorPane and removes all hightlights.
    * @see javax.swing.JEditorPane#setPage(java.net.URL)
    */

   public void setPage(URL url) throws IOException
   {
      super.setPage(url);
      getHighlighter().removeAllHighlights();
   }

   /**
    * Searches the document for the given word and returns the number of
    * matches.
    * @param word
    *           The search word
    * @return int number of matches
    */
   
   public int search(String word)
   {
      int offset = searcher.search(word);
      if (offset != -1)
      {
         lastSearch = word;
         h = getHighlighter().getHighlights();
         index = 0;
         offset = h[index].getStartOffset();
         try
         {
            scrollRectToVisible(modelToView(offset));
         }
         catch (BadLocationException e)
         {
         }
         return h.length;
      }
      return 0;
   }

   /**
    * Search again could be used to select the next highlight and scroll down
    * (or up) to make it visible.
    */
   
   public void searchAgain()
   {
      if (lastSearch == null)
         return;
      if (h != null && h.length > 1)
      {
         index = index + 1 < h.length ? index + 1 : 0;
         int offset = h[index].getStartOffset();
         try
         {
            scrollRectToVisible(modelToView(offset));
         }
         catch (BadLocationException e)
         {
         }
      }
   }
}

/**
 * A basic searcher. It takes a JTextComponent and highlights a search word if
 * found. This class use the String.indexOf() to find matches.
 * @author Thasso
 */

class WordSearcher
{
   /**
    * Create a new searcher for a given JtextComponent
    * @param comp
    *           JTextComponent
    */
   public WordSearcher(JTextComponent comp)
   {
      this.comp = comp;
      this.painter = new MyHighlightPainter(Color.yellow);
   }

   /**
    * Search for a word and return the offset of the first occurrence.
    * Highlights are added for all occurrences found.
    * 
    * @param word
    *           The search word
    * @return int offset from first occurence
    */
   
   public int search(String word)
   {
      int firstOffset = -1;
      Highlighter highlighter = comp.getHighlighter();

      // Remove any existing highlights for last word
      Highlighter.Highlight[] highlights = highlighter.getHighlights();
      for (int i = 0; i < highlights.length; i++)
      {
         Highlighter.Highlight h = highlights[i];
         if (h.getPainter() instanceof MyHighlightPainter)
         {
            highlighter.removeHighlight(h);
         }
      }

      if (word == null || word.equals(""))
      {
         return -1;
      }

      // Look for the word we are given - insensitive search
      String content = null;
      try
      {
         Document d = comp.getDocument();
         content = d.getText(0, d.getLength()).toLowerCase();
      }
      catch (BadLocationException e)
      {
         // Cannot happen
         return -1;
      }

      word = word.toLowerCase();
      int lastIndex = 0;
      int wordSize = word.length();

      while ((lastIndex = content.indexOf(word, lastIndex)) != -1)
      {
         int endIndex = lastIndex + wordSize;
         try
         {
            highlighter.addHighlight(lastIndex, endIndex, painter);
         }
         catch (BadLocationException e)
         {
            // Nothing to do
         }
         if (firstOffset == -1)
         {
            firstOffset = lastIndex;
         }
         lastIndex = endIndex;
      }

      return firstOffset;
   }

   protected JTextComponent comp;

   protected Highlighter.HighlightPainter painter;
}

/**
 * A basic hightlighter.
 * @author Thasso
 */

class MyHighlightPainter extends DefaultHighlighter.DefaultHighlightPainter
{
   public MyHighlightPainter(Color color)
   {
      super(color);
   }
}
