//=================================================================
//                        HTMLViewer
//=================================================================
//
//    This class provides the main access point for setting up the
// requirements for the HTMLView plugin module for the lindyFrame
// framework.
//
//                     << HTMLViewer.java >>
//
//=================================================================
// Copyright (C) 2005-2017 Dana M. Proctor
// Version 1.7 11/10/2017
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version
// 2 of the License, or (at your option) any later version. This
// program is distributed in the hope that it will be useful, 
// but WITHOUT ANY WARRANTY; without even the implied warranty
// of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See
// the GNU General Public License for more details. You should
// have received a copy of the GNU General Public License along
// with this program; if not, write to the Free Software Foundation,
// Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
// (http://opensource.org)
//
//=================================================================
// Revision History
// Changes to the code should be documented here and reflected
// in the present version number. Author information should
// also be included with the original copyright author.
//=================================================================
// Version 1.0 07/27/2013 Initial HTMLViewer Class.
//         1.1 08/02/2013 Added History Functionality and URL Entry
//                        Capability.
//         1.2 08/29/2013 Resources Should NOT Use Local System File Separator
//                        Definition, But Rather Standard File Delimitor Forward
//                        Slash.
//         1.3 06/24/2016 Updated VERSION to Bring Class Into Sync With lindyFrame
//                        v2.5. Now Compiled to Comply With JRE7.
//         1.4 11/07/2016 Updated VERSION & Made Resources Specified in Jar.
//         1.5 11/07/2016 Used imagesDirectory Instead of path as Argument for
//                        Viewer_ToolBar. Updated Version.
//         1.6 04/26/2017 Updated VERSION to Bring Class Into Sync With lindyFrame
//                        v2.7. Now Compiled to Comply With JRE8.
//         1.7 11/10/2017 Comment Changes in Constructor. Updated VERSION to Bring
//                        Class Into Sync With lindyFrame v2.9.
//                           
//-----------------------------------------------------------------
//                 danap@dandymadeproductions.com
//=================================================================

package com.dandymadeproductions.htmlviewer;

import java.awt.BorderLayout;
import java.awt.GridBagConstraints;
import java.awt.GridBagLayout;
import java.awt.Insets;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.KeyAdapter;
import java.awt.event.KeyEvent;
import java.beans.PropertyChangeEvent;
import java.beans.PropertyChangeListener;
import java.io.IOException;
import java.net.MalformedURLException;
import java.net.URL;
import java.util.LinkedList;

import javax.swing.BorderFactory;
import javax.swing.ImageIcon;
import javax.swing.JButton;
import javax.swing.JLabel;
import javax.swing.JMenuBar;
import javax.swing.JPanel;
import javax.swing.JProgressBar;
import javax.swing.JScrollPane;
import javax.swing.JTextField;
import javax.swing.JToolBar;
import javax.swing.SwingUtilities;
import javax.swing.event.HyperlinkEvent;
import javax.swing.event.HyperlinkListener;
import javax.swing.text.html.HTMLEditorKit;

import com.dandymadeproductions.lindyframe.LindyFrame;
import com.dandymadeproductions.lindyframe.gui.Main_Frame;
import com.dandymadeproductions.lindyframe.utilities.LindyFrame_ResourceBundle;
import com.dandymadeproductions.lindyframe.utilities.LindyFrame_Utils;

/**
 *    The HTMLViewer class provides the main access point for
 * setting up the requirements for the HTMLView plugin module
 * for the lindyFrame framework.
 * 
 * @author Dana M. Proctor
 * @version 1.7 11/10/2017
 */

class HTMLViewer implements ActionListener
{
   // Class Instances
   private ImageIcon tabIcon;
   
   private GridBagLayout gridbag;
   private GridBagConstraints constraints;

   private JPanel mainPanel, urlEntryPanel, statusPanel;
   private JButton historyBackButton, historyForwardButton, refreshButton;
   private JTextField urlEntryTextField;
   private LinkedList<String> historyList;
   protected HTMLEditorPane viewerPane;
   private JProgressBar progressBar;
   private JLabel statusLabel;

   private Viewer_MenuBar menuBar;
   private Viewer_ToolBar toolBar;

   private LindyFrame_ResourceBundle resourceBundle;
   
   private String loadingPageResource, pageNotFoundResource;
   private int stateHistoryIndex;

   private final static String VERSION = "Version 1.7";
   private final static String DESCRIPTION = "The HTMLViewer plugin provides a mean to demostrate a module"
                                             + " example for the lindyFrame framework. The plugin is a basic"
                                             + " HTML viewer that displays the tutorial for creating plugins"
                                             + " with lindyFrame.";
   
   private static final int STATE_HISTORY_LIMIT = 25;

   //==============================================================
   // HTMLViewer Constructor
   //==============================================================

   public HTMLViewer(Main_Frame parent, String path)
   {
      // Constructor Instances.
      String pathDirectory, localeDirectory, imagesDirectory;
      MenuActionListener pluginMenuListener;

      // Setup the Main panel and the plugin's components.

      mainPanel = new JPanel(new BorderLayout());

      // ** NOTE: file/local network only, locale resource not in JAR **
      // pathDirectory = path + "/" + "HTMLViewer" + "/";
      // localeDirectory = "locale/";
      // imagesDirectory = "images/";

      // ** NOTE: file/local network/http/ftp locale resource in JAR **
      pathDirectory = path + "/" + "HTMLViewer.jar";
      localeDirectory = "lib/plugins/HTMLViewer/locale/";
      imagesDirectory = "lib/plugins/HTMLViewer/images/";

      resourceBundle = new LindyFrame_ResourceBundle(pathDirectory, LindyFrame.getDebug());
      resourceBundle.setLocaleResource(localeDirectory, "HTMLViewer", LindyFrame.getLocaleString());

      tabIcon = resourceBundle.getResourceImage(imagesDirectory + "icons/htmlViewerIcon.png");
      loadingPageResource = resourceBundle.getResourceString("HTMLViewer.label.LoadingPage", "Loading Page");
      pageNotFoundResource = resourceBundle.getResourceString("HTMLViewer.label.PageNotFound",
                                                              "Page Not Found");
      
      // Create the GUI components.
      
      gridbag = new GridBagLayout();
      constraints = new GridBagConstraints();

      // URL Entry Panel.
      initURLEntryPanel(imagesDirectory);
      mainPanel.add(urlEntryPanel, BorderLayout.NORTH);

      // HTML Panel.
      initViewer();
      mainPanel.add(new JScrollPane(viewerPane), BorderLayout.CENTER);

      // Status Panel
      initStatusPanel();
      mainPanel.add(statusPanel, BorderLayout.SOUTH);

      // Setup the MenuBar and ToolBar to be used by the plugin.

      pluginMenuListener = new MenuActionListener(parent, this, resourceBundle);
      menuBar = new Viewer_MenuBar(parent, resourceBundle, pluginMenuListener);
      toolBar = new Viewer_ToolBar("HTMLViewer ToolBar", parent, imagesDirectory, resourceBundle,
                                   pluginMenuListener);
      
      loadPage("file:"
         + LindyFrame_Utils.getResourceBundle().getResourceFile(
            "docs/Plugins/Tutorial/LindyFrame_PluginTutorial.html").getPath());
   }

   //==============================================================
   // Class method to create the URL entry components.
   //==============================================================

   private void initURLEntryPanel(String imagesDirectory)
   {
      // Method Instances
      JPanel historyPanel, refreshPanel;
      ImageIcon historyBackIcon, historyForwardIcon, refreshIcon;
      
      urlEntryPanel = new JPanel(gridbag);
      urlEntryPanel.setBorder(BorderFactory.createCompoundBorder(
         BorderFactory.createEmptyBorder(8, 8, 8, 8), BorderFactory.createEtchedBorder()));
      
      // History Componenets
      
      historyPanel = new JPanel();
      historyPanel.setBorder(BorderFactory.createEmptyBorder(0, 0, 0, 0));
      
      historyBackIcon = resourceBundle.getResourceImage(imagesDirectory + "icons/historyBackIcon.png");
      historyBackButton = new JButton(historyBackIcon);
      historyBackButton.setMargin(new Insets(0, 0, 0, 0));
      historyBackButton.setEnabled(false);
      historyBackButton.addActionListener(this);
      historyPanel.add(historyBackButton);
      
      historyForwardIcon = resourceBundle.getResourceImage(imagesDirectory + "icons/historyForwardIcon.png");
      historyForwardButton = new JButton(historyForwardIcon);
      historyForwardButton.setMargin(new Insets(0, 0, 0, 0));
      historyForwardButton.setEnabled(false);
      historyForwardButton.addActionListener(this);
      historyPanel.add(historyForwardButton);
      
      LindyFrame_Utils.buildConstraints(constraints, 0, 0, 1, 1, 2, 100);
      constraints.fill = GridBagConstraints.BOTH;
      constraints.anchor = GridBagConstraints.CENTER;
      gridbag.setConstraints(historyPanel, constraints);
      urlEntryPanel.add(historyPanel);
      
      // URL Entry Component
      
      urlEntryTextField = new JTextField("");
      urlEntryTextField.setBorder(BorderFactory.createCompoundBorder(
         BorderFactory.createEmptyBorder(4, 2, 4, 2), BorderFactory.createLoweredBevelBorder()));
      urlEntryTextField.addMouseListener(LindyFrame.getPopupMenuListener());
      
      LindyFrame_Utils.buildConstraints(constraints, 1, 0, 1, 1, 97, 100);
      constraints.fill = GridBagConstraints.BOTH;
      constraints.anchor = GridBagConstraints.CENTER;
      gridbag.setConstraints(urlEntryTextField, constraints);
      urlEntryPanel.add(urlEntryTextField);
      
      // Refresh URL Component
      
      refreshPanel = new JPanel();
      refreshPanel.setBorder(BorderFactory.createEmptyBorder(0, 0, 0, 0));
      
      refreshIcon = resourceBundle.getResourceImage(imagesDirectory + "icons/refreshIcon.png");
      refreshButton = new JButton(refreshIcon);
      refreshButton.setMargin(new Insets(0, 0, 0, 0));
      refreshButton.addActionListener(this);
      refreshPanel.add(refreshButton);

      LindyFrame_Utils.buildConstraints(constraints, 2, 0, 1, 1, 1, 100);
      constraints.fill = GridBagConstraints.BOTH;
      constraints.anchor = GridBagConstraints.CENTER;
      gridbag.setConstraints(refreshPanel, constraints);
      urlEntryPanel.add(refreshPanel);
      
      // Inner Text Field Enter Key Listener Class.
      
      historyList = new LinkedList <String>();
      stateHistoryIndex = -1;
      
      urlEntryTextField.addKeyListener(new KeyAdapter()
      {
        String lastURL = "";
        
        public void keyReleased(KeyEvent e)
        {
           if(e.getKeyCode() == KeyEvent.VK_ENTER)
           {
              if(!urlEntryTextField.getText().trim().equals(lastURL))
              {
                 lastURL = urlEntryTextField.getText().trim();
                 // System.out.println("HTMLViewer.urlEntryTextField.KeyListener lastURL: " + lastURL);
                 historyList.add(lastURL);
                 
                 historyBackButton.setEnabled(true);
                 stateHistoryIndex++;
              }
              else
                 return;
              
              updateViewer(urlEntryTextField.getText());
           }
        }     
     });   
   }

   //==============================================================
   // Class method to create the HTML viewer editor pane.
   //==============================================================

   private void initViewer()
   {
      viewerPane = new HTMLEditorPane();
      viewerPane.setMargin(new Insets(5, 5, 5, 5));
      viewerPane.setEditorKit(new HTMLEditorKit());
      viewerPane.setEditable(false);
      
      viewerPane.addHyperlinkListener(new HyperlinkListener()
      {
         public void hyperlinkUpdate(HyperlinkEvent e)
         {
            if (e.getEventType() == HyperlinkEvent.EventType.ACTIVATED)
            {
               updateViewer(e.getURL());
            }
         }

      });
      
      viewerPane.addPropertyChangeListener(new PropertyChangeListener()
      {

         public void propertyChange(PropertyChangeEvent evt)
         {
            if (evt.getPropertyName().equals("page"))
            {
               setStatus(null);
            }
         }
      });
   }

   //==============================================================
   // Class method to initialize the status indicator at the bottom
   // of the module that will be used to indicate current page
   // loads.
   //==============================================================

   private void initStatusPanel()
   {
      statusPanel = new JPanel(gridbag);
      statusPanel.setBorder(BorderFactory.createCompoundBorder(BorderFactory.createEmptyBorder(4, 4, 4, 4),
         BorderFactory.createLoweredBevelBorder()));

      progressBar = new JProgressBar();
      progressBar.setBorder(BorderFactory.createRaisedBevelBorder());

      LindyFrame_Utils.buildConstraints(constraints, 0, 0, 1, 1, 10, 100);
      constraints.fill = GridBagConstraints.HORIZONTAL;
      constraints.anchor = GridBagConstraints.WEST;
      gridbag.setConstraints(progressBar, constraints);
      statusPanel.add(progressBar);

      statusLabel = new JLabel();

      LindyFrame_Utils.buildConstraints(constraints, 1, 0, 1, 1, 90, 100);
      constraints.fill = GridBagConstraints.HORIZONTAL;
      constraints.anchor = GridBagConstraints.CENTER;
      gridbag.setConstraints(statusLabel, constraints);
      statusPanel.add(statusLabel);
   }
   
   //==============================================================
   // ActionEvent Listener method for detecting the inputs from
   // the class, buttons.
   //==============================================================
   
   public void actionPerformed(ActionEvent evt)
   {
      Object panelSource = evt.getSource();

      if (panelSource instanceof JButton)
      {
         if (panelSource == historyBackButton || panelSource == historyForwardButton)
         {   
            executeHistoryAction(panelSource);
         }
         else if (panelSource == refreshButton)
         {
            setLocation(urlEntryTextField.getText().trim());
         }     
      }
   }
   
   //==============================================================
   // Class method to control the history indexing and loading.
   //==============================================================

   private void executeHistoryAction(Object panelSource)
   {
      // Method Instances.
      
      // Back History Action
      if (panelSource == historyBackButton)
      {
         // Decrement and check lower bound.
         stateHistoryIndex--;
         if (stateHistoryIndex <= 0)
         {
            stateHistoryIndex = 0;
            historyBackButton.setEnabled(false);
         }
      }
      
      // Forward History Action
      if (panelSource == historyForwardButton)
      {
         // Increment and check upper bound.
         stateHistoryIndex++;
         if (stateHistoryIndex > (STATE_HISTORY_LIMIT - 1))
         {
            stateHistoryIndex = (STATE_HISTORY_LIMIT - 1);
            historyForwardButton.setEnabled(false);
         }
      }
      
      // Check Overall movement of index against
      // history buttons and bound as needed.
      // System.out.println("HTMLViewer.executeHistory() stateHistoryIndex: " + stateHistoryIndex);
      
      if (stateHistoryIndex == 0)
         historyBackButton.setEnabled(false);
      else
         historyBackButton.setEnabled(true);
      
      if (stateHistoryIndex < (historyList.size() - 1))
         historyForwardButton.setEnabled(true);
      else
         historyForwardButton.setEnabled(false);
      
      // Load the history selection.
      if (panelSource != null)
         setLocation(historyList.get(stateHistoryIndex));
   }
   
   //==============================================================
   // Class load the URL location provided.
   //==============================================================

   private boolean setLocation(String prefix)
   {
      setStatus(" Loading Page " + prefix);
      progressBar.setIndeterminate(true);

      if (prefix.endsWith("/"))
      {
         prefix += "index.html";
      }

      final String pref = prefix;
      
      SwingUtilities.invokeLater(new Runnable()
      {
         public void run()
         {
            updateViewer(pref);
         }
      });

      /*
      SwingUtilities.invokeLater(new Runnable()
      {
         public void run()
         {
            setStatus("Updating");
            progressBar.setIndeterminate(false);
            setStatus(null);
         }
      });
      */
      return true;
   }

   //==============================================================
   // Class method to load a URL into the HTML editor viewer pane.
   //==============================================================

   public boolean updateViewer(String url)
   {
      try
      {
         return updateViewer(new URL(url));
      }
      catch (MalformedURLException e)
      {
         System.err.println("Given url is not valid " + url);
         return false;
      }
   }

   public boolean updateViewer(final URL url)
   {
      try
      {
         setStatus(loadingPageResource + ": " + url.toExternalForm());
         viewerPane.setPage(url);
      }
      catch (IOException e)
      {
         if (LindyFrame.getDebug())
            System.out.println("HTMLViewer.updateViewer() Could Not Load Page: " + url);

         setStatus(pageNotFoundResource + ": " + url.toExternalForm());

         progressBar.setIndeterminate(false);
         progressBar.setVisible(false);
         return false;
      }
      return true;
   }
   
   //==============================================================
   // Class method to set the status panel indicator for activity.
   //==============================================================

   private void setStatus(String text)
   {
      setStatus(text, text == null ? -1 : 101);
   }

   public void setStatus(String string, int i, final int seconds)
   {
      setStatus(string, i);

      Thread t = new Thread(new Runnable()
      {
         public void run()
         {
            try
            {
               Thread.sleep(seconds * 1000);
            }
            catch (InterruptedException e)
            {
               e.printStackTrace();
            }
            setStatus(null);
         }
      });
      t.start();
   }

   private void setStatus(String text, int value)
   {
      statusLabel.setText(text == null ? " " : text);

      if (value < 0)
      {
         progressBar.setIndeterminate(false);
         progressBar.setVisible(false);
      }
      else if (value > 100)
      {
         progressBar.setIndeterminate(true);
         progressBar.setVisible(true);
      }
      else
      {
         progressBar.setVisible(true);
         progressBar.setValue(value);
      }
   }
   
   //==============================================================
   // Class method to load a page view opening file or directly.
   //==============================================================

   protected void loadPage(String urlString)
   {
      if (setLocation(urlString))
      {
         historyList.add(urlString);
         urlEntryTextField.setText(urlString);
         stateHistoryIndex++;
         executeHistoryAction(null);
      }
   }
   
   //==============================================================
   // Class method to to the plugin's JMenuBar
   //==============================================================

   protected JMenuBar getMenuBar()
   {
      return menuBar;
   }

   //==============================================================
   // Class method get the plugin's JToolBar
   //==============================================================

   protected JToolBar getToolBar()
   {
      return toolBar;
   }

   //==============================================================
   // Class method to get the main panel associated with the plugin.
   //==============================================================

   protected JPanel getPanel()
   {
      return mainPanel;
   }

   //==============================================================
   // Class method to get the plugin's version.
   //==============================================================

   protected String getVersion()
   {
      return VERSION;
   }

   //==============================================================
   // Class method to get the plugin's description.
   //==============================================================

   protected String getDescription()
   {
      return DESCRIPTION;
   }

   //==============================================================
   // Class method to get the icon that will be used in the
   // lindyFrame tab.
   //==============================================================

   protected ImageIcon getTabIcon()
   {
      return tabIcon;
   }
}
